/*
 * @brief Secondary loader SPI host interface handler
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2014
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

#include "chip.h"
#include "sl_common.h"

/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/

static SPI_Type *savedPort;
static SPI_Type *pSPIArray[1] = {
    SPI0
};

extern uint8_t Chip_SPIS_FindSSEL(SPI_Type *pSPI, uint32_t data);

/* Possible function selections for SPI mapped pins in IOCON */
#define SPIPURP_SCK         0
#define SPIPURP_MOSI        1
#define SPIPURP_MISO        2
#define SPIPURP_SSEL        3
#define SPIPURP_MAPMAPPINS  17

typedef struct {
    uint8_t     spiNum;            /* SPin number, 0 or 1 */
    uint8_t     port;              /* Port */
    uint8_t     pin;               /* Pin */
    uint8_t     func;              /* IOCON function number */
    uint8_t     purpose;           /* 0 = SCK, 1 = MOSI, 2 = MISO, 3 = SSEL */
} SL_SPIFNUCMAP_T;


static const SL_SPIFNUCMAP_T spiMappedFuncs[SPIPURP_MAPMAPPINS] = {
};

struct SPIS_XFER;
typedef void (*SPISlaveXferCSAssert)(struct SPIS_XFER *pSlaveXfer);
typedef void (*SPISlaveXferSend)(struct SPIS_XFER *pSlaveXfer);
typedef void (*SPISlaveXferRecv)(struct SPIS_XFER *pSlaveXfer);
typedef void (*SPISlaveXferCSDeAssert)(struct SPIS_XFER *pSlaveXfer);

typedef struct {
    SPISlaveXferCSAssert    slaveXferCSAssert;        /** SPI transfer start callback, called on SPI CS assertion */
    SPISlaveXferSend        slaveXferSend;            /** SPI transfer data receive buffer callback, called when a receive buffer is needed */
    SPISlaveXferRecv        slaveXferRecv;            /** SPI transfer send buffer callback, called when data is needed */
    SPISlaveXferCSDeAssert  slaveXferCSDeAssert;      /** SPI transfer completion callback, called on SPI CS deassertion */
} SPIS_CALLBACKS_T;

/** Slave transfer data context */
typedef struct SPIS_XFER {
    const SPIS_CALLBACKS_T *pCB;       /** Pointer to SPI slave callback functions */
    union {                            /** Pointer to receive buffer, set to NULL to toss receeive data */
        uint8_t *pRXData8;             /** Receive buffer used with data transfer size <= 8-bits, modified by driver */
        uint16_t *pRXData16;           /** Receive buffer used with data transfer size > 8-bits, modified by driver */
    };

    union {                            /** Pointer to transmit buffer, set to NULL to transmit 0x0 */
        uint8_t *pTXData8;             /** Send buffer used with data transfer size <= 8-bits, modified by driver */
        uint16_t *pTXData16;           /** Send buffer used with data transfer size > 8-bits, modified by driver */
    };

    uint16_t rxCount;                  /** Size of the pRXData buffer in items (not bytes), modified by driver */
    uint16_t txCount;                  /** Number of items (not bytes) to send in pTXData buffer, modified by driver */
    uint16_t dataRXferred;             /** Total items (not bytes) received, modified by driver */
    uint16_t dataTXferred;             /** Total items (not bytes) transmitted, modified by driver */
    uint8_t sselNum;                   /** Slave number assigned to this transfer, 0 - 3, modified by driver */
} SPIS_XFER_T;


/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/

volatile SPIS_XFER_T spiSlaveXfer;
extern int buffXferPending;
extern uint32_t g_spiTxPending;
int spi_tx_index=0;

/* DMA descriptors must be aligned to 16 bytes */
#if defined(__CC_ARM)    /* Keil support */

#elif defined(__ICCARM__)    /* IAR support */
#pragma data_alignment=16
static DMA_CHDESC_T dmaSPISTxDesc[1];
#pragma data_alignment=16
static DMA_CHDESC_T dmaSPISRxDesc[1];
#elif defined( __GNUC__ )    /* GCC support */

#endif /* defined (__GNUC__) */

/*****************************************************************************
 * Private functions
 ****************************************************************************/

#define SPI_READDMA_XFERSIZE    (SL_HOSTIFBUFFSIZE + 1)

/* SPI slave select assertion callback function */
static void SPISlaveAssert(spi_transfer_t *pSlaveXfer)
{
    Hostif_DeAssertIRQ();
}

/* SPI slave send data callback function */
static void SPISlaveSendData(spi_transfer_t *pSlaveXfer)
{
    /* Do nothing */
}

/* SPI slave receive data callback function */
static void SPISlaveRecvData(spi_transfer_t *pSlaveXfer)
{
    /* Do nothing */
}

/* SPI slave select de-assertion callback function */
static void SPISlaveDeAssert(spi_transfer_t *pSlaveXfer)
{
    /* check if we sending response */
    if (g_spiTxPending) {
        startHostIfSPI();
        spi_tx_index++;
    }
    else {

    }
    //Indicate a data packet received.
    buffXferPending = 1;
}

/* SPI slave driver callbacks */
static const SPIS_CALLBACKS_T spiSlaveCallbacks = {
    (SPISlaveXferCSAssert) &SPISlaveAssert,
    (SPISlaveXferSend) &SPISlaveSendData,
    (SPISlaveXferRecv) &SPISlaveRecvData,
    (SPISlaveXferCSDeAssert) &SPISlaveDeAssert
};

/* Maps a pin's IOCON function to it's SPI number, port, pin, and purpose */
static void spiMapPin(int spiNum, SL_PORTPIN_T *pSpiPortPin, uint8_t purp)
{
    int i = 0;
    uint32_t func;
    bool mapped = false;

    while ((mapped == false) && (i < SPIPURP_MAPMAPPINS)) {
        if (spiMappedFuncs[i].spiNum == spiNum) {
            if (spiMappedFuncs[i].purpose == purp) {
                if (pSpiPortPin->port == spiMappedFuncs[i].port) {
                    if (pSpiPortPin->pin == spiMappedFuncs[i].pin) {
                        /* Matched function to pin */
                        func = (uint32_t) spiMappedFuncs[i].func;

                        break;
                    }
                }
            }
        }

        i++;
    }
}

/*****************************************************************************
 * Public functions
 ****************************************************************************/

void setupMuxingSPI(int spiNum)
{
    /* Pin selection based on port. Note MISO is not setup here, as it is
       setup later after A5 packet processing (also ok for app provided MISO). */
    spiMapPin(spiNum, &slIfConfig.hostMosiPortPin, SPIPURP_MOSI);
    spiMapPin(spiNum, &slIfConfig.hostSselPortPin, SPIPURP_SSEL);
    spiMapPin(spiNum, &slIfConfig.hostSckPortPin, SPIPURP_SCK);
}

/* Sets up pinmuxing for an SPI MISO pin only */
void setupMuxingSPIMiso(int spiNum)
{
    spiMapPin(spiNum, &slIfConfig.hostMisoPortPin, SPIPURP_MISO);
}

void setupInterfaceSPI(int spiNum)
{
    savedPort = pSPIArray[0];

    /* Setup slave controller SSEL for active low select */
    SPI0->CFG = ~(1<<8) & (SPI0->CFG & 0x1BD);

    /* Setup slave controller for 8-bit transfers */
    SPI0->TXCTL = 0 | (((8 - 1) & 0x0F) << 24);

    /* Enable SPI0 */
    SPI0->CFG |= 0x01;

    SPI0->INTENSET = SPI_INTENSET_RXRDYEN_MASK | \
                       SPI_INTENSET_RXOVEN_MASK  | \
                     SPI_INTENSET_TXUREN_MASK  | \
                       SPI_INTENSET_SSAEN_MASK   | \
                     SPI_INTENSET_SSDEN_MASK;

    /* Setup slave transfer callbacks in the transfer descriptor */
    spiSlaveXfer.pCB = &spiSlaveCallbacks;
}

void startHostIfSPI(void)
{    
    /*pTXData8 point to sendBuff.buffer[1], because sendBuff.buffer[0] will be preload to TXData now*/
    spiSlaveXfer.pTXData8 = &sendBuff.buffer[1];
    spiSlaveXfer.txCount = SL_HOSTIFBUFFSIZE;
    spiSlaveXfer.pRXData8 = &recvBuff.buffer[0];
    spiSlaveXfer.rxCount = SL_HOSTIFBUFFSIZE;
    spiSlaveXfer.dataRXferred = 0;
    spiSlaveXfer.dataTXferred = 0;

    /* Flush any pre loaded SPI data */
    //Chip_SPI_FlushFifos(LPC_SPI0);
    SPI_Enable(SPI0, false);
    SPI_Enable(SPI0, true);
    
    /*Preload data to TXDATA to response master*/
    SPI_WriteData(SPI0, sendBuff.buffer[0]);

}

uint8_t SPI_findSSEL(SPI_Type *pSPI, uint32_t data)
{
    int i;
    uint8_t ssel = 0;

    for (i = 0; i <= 3; i++) {
        if ((data & (1 << ((i) + 16))) == 0) {
            /* Signal is active on low */
            ssel = (uint8_t) i;
        }
    }

    return ssel;

}

/* SPI slave transfer state change handler */
void SPI_Slave_Rx_Blocking(SPI_Type *pSPI, SPIS_XFER_T *xfer)
{

    /* waiting for the rising edge of the WS signal */
    while((SPI_GetStatusFlags(pSPI) & SPI_STAT_SSD_MASK) == 0)
    {
        while ((SPI_GetStatusFlags(pSPI) & SPI_STAT_RXRDY_MASK) != 0) {
            /* Get raw data and status */
            *xfer->pRXData8  = SPI_ReadData(pSPI);
            xfer->pRXData8++;                    
            xfer->dataRXferred++;        
        }
    }
}

void SPI_Slave_Tx_Blocking(SPI_Type *pSPI, SPIS_XFER_T *xfer)
{
    uint32_t  data;

    /* waiting for the rising edge of the WS signal */
    while((SPI_GetStatusFlags(pSPI) & SPI_STAT_SSD_MASK) == 0)
    {
        while ((SPI_GetStatusFlags(pSPI) & SPI_STAT_TXRDY_MASK) != 0) {
            data = (uint32_t) *xfer->pTXData8;
            SPI_WriteData(pSPI, data);
            xfer->pTXData8++;
            xfer->dataTXferred++;                                
        }        
    }                    
}

/* SPI slave transfer state change handler */
uint32_t SPI_Slave_Handler(SPI_Type *pSPI, SPIS_XFER_T *xfer)
{
    uint32_t staterr;
    staterr = SPI_GetStatusFlags(pSPI) & (SPI_STAT_RXOV_MASK | SPI_STAT_TXUR_MASK);
    if (staterr != 0) {
        SPI_ClearStatusFlags(pSPI, staterr);
    }

    /* Slave assertion, CS pin is pulled low */
    if ((SPI_GetStatusFlags(pSPI) & SPI_STAT_SSA_MASK) != 0) {
        SPI_ClearStatusFlags(pSPI, SPI_STAT_SSA_MASK);

        /* Determine SPI select. Read the data FIFO to get the slave number. Data
           should not be in the receive FIFO yet, only the statuses */
        xfer->sselNum = SPI_findSSEL(pSPI, SPI_ReadData(pSPI));

        /* SSEL assertion callback, if CS is pulled low, pull IRQ high */
        xfer->pCB->slaveXferCSAssert(xfer);
        
        /*If CS pin is low and g_spiTxPending==0, receive data */
        if(g_spiTxPending == 0)
        {
            xfer->dataRXferred = 0;    
            SPI_Slave_Rx_Blocking(SPI0,xfer);
        }
        
        /*Response data to SPI master*/
        if(g_spiTxPending == 1)
        {
            xfer->dataTXferred = 0;
            SPI_Slave_Tx_Blocking(SPI0,xfer);
            g_spiTxPending = 0;
        }
    }

    /* Slave de-assertion, if the rising edge of CS is detected---receive data completed or response data completed*/
    if ((SPI_GetStatusFlags(pSPI) & SPI_STAT_SSD_MASK) != 0) {
        SPI_ClearStatusFlags(pSPI, SPI_STAT_SSD_MASK);
       
        /*startHostSPI, configure spiSlaveXfer structure for new command or new response*/
        xfer->pCB->slaveXferCSDeAssert(xfer);
    }
    return staterr;

}

/* SPI host interface processing loop */
void loopHostIfSPI(int spiNum)
{
    SPI_Slave_Handler(SPI0, (SPIS_XFER_T *)&spiSlaveXfer);
    recvBuff.bytes = spiSlaveXfer.dataRXferred;
}

/* Shut down a SPI slave interface */
void shutdownInterfaceSPI(int spiNum)
{
    SPI_Deinit(SPI0);
}
